package integer;


import java.awt.Color;
import java.util.ArrayList;
import java.util.List;
import net.useobjects.AbstractGroup;
import net.useobjects.Circle;
import net.useobjects.Line;
import net.useobjects.Rectangle;
import net.useobjects.mouse.MouseAdapter;
import net.useobjects.mouse.MouseEvent;

//Slider meni ciselnu hodnotu v urcitom intervale
public class Slider extends AbstractGroup implements BoundedIntegerEventSource {

    private Circle button; //bezec
    private int mouseOffset; //rozdiel medzi x-ovou suradnicou posuvnika a x-ovou suradnico mysi
    private BoundedIntegerModel model; //model aktualnej hodnoty (polohy) slidera
    private List<BoundedIntegerListener> listeners; //zoznam listenerov 


    /*
     * value - inicializacna hodnota
     * min - minimalna hodnota
     * max - maximalna hodnota
     * parentGroup - skupina do ktorej bude slider patrit
     * positionX - x-ova suradnica polohy slidera
     * positionY - y-ova suradnica polohy slidera
     * buttonColor - farba bezca
     */
    public Slider(int value, int min, int max, AbstractGroup parentGroup, double positionX, double positionY, Color buttonColor) {
        this(new BoundedIntegerModel(value, min, max), parentGroup, positionX, positionY, buttonColor);
    }
    
    public Slider(BoundedIntegerModel model, AbstractGroup parentGroup, double positionX, double positionY, Color buttonColor) {
        super(parentGroup, positionX, positionY);
        this.model = model;
        listeners = new ArrayList<BoundedIntegerListener>();

        new Rectangle(this, 0, 0, model.getRangeLength() + 20, 20, 0, Color.LIGHT_GRAY, true); //background
        new Line(this, 0, 0, 0, Color.BLACK, model.getRangeLength()); //line
        button = new Circle(this, 0, 0, 5, buttonColor, true); //bezec
        setButtonPosition(model); //nastavime poziciu bezca podla aktualneho stavu modelu
        
        model.addBoundedIntegerListener(new IntegerObserver());

        button.addMouseListener(new ValueController() );
    }


    private class IntegerObserver implements BoundedIntegerListener {
        @Override
        public void integerChanged(BoundedIntegerEvent event) {
            valueChanged(event);
        }
    }
    
    //Interpretuje akcie pouivatela (vykonane na view), transformuje ich na zmeny v modely
    private class ValueController extends MouseAdapter { //Controller (zmena polohy bezca -> zmena ciselnej hodnoty)
        @Override
        public void onMousePressed(MouseEvent e) {
            mouseOffset = (int) e.getX();
        }
        @Override
        public void onMouseDragged(MouseEvent e) {
            //neposunieme bezec priamo, ale zmenime hodnotu v modely, 
            //potom model oznamy slider-u zmenu, slider na nu zareaguje zmenou pozicie bezca
            int newValue = (int)(model.getRangeLength()/2 + model.getMin() + button.getPositionX() + e.getX() - mouseOffset);
            setValue(newValue);
        }
    }

    public BoundedIntegerModel getModel() {
        return model;
    }
    
    public int getValue() {
        return model.getValue();
    }

    public int getRangeLength() {
        return model.getRangeLength();
    }

    public void setValue(int newValue) {
        model.setValue(newValue); //ak sa zmeni hodnota, tak model oznami zmenu a zavola sa metoda valueChanged
    }

    private void setButtonPosition(BoundedIntegerModel model) {
        button.setPositionX(-model.getRangeLength() / 2 + model.getValue() - model.getMin());
    }
    
    private void valueChanged(BoundedIntegerEvent event) {
        setButtonPosition((BoundedIntegerModel)event.getSource());
        notifyListeners(event);
    }

    public void addBoundedIntegerListener(BoundedIntegerListener listener) {
        listeners.add(listener);
    }
    
    public void removeBoundedIntegerListener(BoundedIntegerListener listener) {
        listeners.remove(listener);
    }

    private void notifyListeners(BoundedIntegerEvent inputEvent) {
        //Preposleme udalost dalej, ale ako zdroj nastavime this
        //IntegerEvent je immutable, takze staci vytvorit jednu instanciu
        BoundedIntegerEvent outputEvent = new BoundedIntegerEvent(this, inputEvent.getValue(), inputEvent.getMin(), inputEvent.getMax()); 
        for (BoundedIntegerListener listener : listeners) {
            listener.integerChanged(outputEvent);
        }
    }
}
