package color;


import java.awt.Color;
import java.util.ArrayList;
import java.util.List;

//Trieda ColorModel reprezentuje udaje o farbe
public class ColorModel {
    private int red;   //cervena zlozka farby
    private int green; //zelena zlozka farby
    private int blue;  //modra zlozka farby
    
    private List<ColorListener> listeners; //zoznam listnerov, ktorym sa posielaju informacie o zmene farby

    public ColorModel(int red, int green, int blue) {
        checkAndSetRed(red);
        checkAndSetGreen(green);
        checkAndSetBlue(blue);
        
        listeners = new ArrayList<ColorListener>();
    }
    
    public ColorModel(Color color) {
        this(color.getRed(), color.getGreen(), color.getBlue());
    }
    
    public int getRed() {
        return red;
    }
    
    public int getGreen() {
        return green;
    }
    
    public int getBlue() {
        return blue;
    }
    
    public Color getColor() {
        return new Color(red, green, blue);
    }

    //skontroluje, ci je zlozka farby v rozsahu <0, 255>. Ak nie je tak vyhodi vynimku
    private void checkComponentRange(int componentValue, String componentName) {
        final int MIN = getMinComponentValue();
        final int MAX = getMaxComponentValue();
        if( componentValue < MIN || componentValue > MAX) {
            throw new IllegalArgumentException(componentName + " zlozka " + componentValue + " je mimo rozsahu <" + MIN + ", " + MAX +">)");
        }
    }
    
    //skontroluje ci je hodnota farebnej zlozky v spravnom rozsahu a nastavi novu hodnotu atribudu pre cervenu zlozku
    private void checkAndSetRed(int newRed) {
        checkComponentRange(newRed, "cervena");
        this.red = newRed;
    }

    //skontroluje ci je hodnota farebnej zlozky v spravnom rozsahu a nastavi novu hodnotu atribudu pre zelenu zlozku
    private void checkAndSetGreen(int newGreen) {
        checkComponentRange(newGreen, "zelena");
        this.green = newGreen;
    }

    //skontroluje ci je hodnota farebnej zlozky v spravnom rozsahu a nastavi novu hodnotu atribudu pre modru zlozku
    private void checkAndSetBlue(int newBlue) {
        checkComponentRange(newBlue, "modra");
        this.blue = newBlue;
    }
    
    public void setRed(int red) {
        if( getRed() != red ) { //udalost generujem, iba ak sa hodnota farebnej zlozky meni
            checkAndSetRed(red);
            notifyListeners(true, false, false);
        }
    }
    
    public void setGreen(int green) {
        if( getGreen() != green ) {
            checkAndSetGreen(green);
            notifyListeners(false, true, false);
        }
    }

    public void setBlue(int blue) {
        if( getBlue() != blue ) {
            checkAndSetBlue(blue);
            notifyListeners(false, false, true);
        }
    }
    
    public void setColor(int red, int green, int blue) {
        boolean redChanged   = (getRed()   != red);
        boolean greenChanged = (getGreen() != green);
        boolean blueChanged  = (getBlue()  != blue);
        
        if( redChanged || greenChanged || blueChanged ) { //udalost generujeme iba ak sa farba meni
            checkAndSetRed(red);
            checkAndSetGreen(green);
            checkAndSetBlue(blue);
            notifyListeners(redChanged, greenChanged, blueChanged);
        }
    }
    
     public void setColor(Color color) {
         setColor(color.getRed(), color.getGreen(), color.getBlue());
     }

    //pridanie listnenera do zoznamu listenerov, ktorym sa posielaju informacie o zmene farby 
    public void addColorListener(ColorListener listener) {
        listeners.add(listener);
    }
    
    //vyber listenera zo zoznamu listenerov, ktorym sa posielaju informacie o zmene farby
    public void removeColorListener(ColorListener listener) {
        listeners.remove(listener);
    }
    
    //oznamenie zmeny farby vsetkym listenerom
    private void notifyListeners(boolean redChanged, boolean greenChanged, boolean blueChanged) {
        ColorEvent colorEvent = new ColorEvent(this, red, green, blue, redChanged, greenChanged, blueChanged); //staci vytvorit iba raz, kedze je immutable
        for(ColorListener listener: listeners) {
            listener.colorChanged(colorEvent);
        }
    }
    
    //Vrati maximalnu hodnotu farebnej zlozky
    public int getMaxComponentValue() {
        return 255;
    }
    
    //Vrati minimalnu hodnotu farebnej zlozky
    public int getMinComponentValue() {
        return 0;
    }
}
